<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Coupon;
use Carbon;
use App\Cart;
use Auth;
use Session;
use DB;

class ApplyCouponController extends Controller
{
    public function applycoupon(Request $request)
    {
        $coupon = Coupon::where('code', $request->coupon)->first();
        $mytime = Carbon\Carbon::now();
        $date = $mytime->toDateTimeString();

        if (isset($coupon)) {
            if ($coupon->expirydate >= $date) {
                if ($coupon->maxusage != 0) {
                    if ($coupon->link_by == 'course') {
                        return $this->validCouponForCourse($coupon);
                    } elseif ($coupon->link_by == 'cart') {
                        return $this->validCouponForCart($coupon);
                    } elseif ($coupon->link_by == 'category') {
                        return $this->validCouponForCategory($coupon);
                    } elseif ($coupon->link_by == 'bundle') {
                        return $this->validCouponForBundle($coupon);
                    }
                } else {
                    return back()->with('fail', 'Coupon max limit reached !');
                }
            } else {
                return back()->with('fail', 'Coupon Expired !');
            }
        } else {
            return back()->with('fail', 'Invalid Coupon !');
        }
    }

    protected function calculateDiscount($price, $coupon) 
    {
        if ($coupon->distype == 'per') {
            return $price * $coupon->amount / 100;
        }
        return $coupon->amount;
    }

    protected function validateDiscountAmount($totalPrice, $discountAmount) 
    {
        if ($discountAmount >= $totalPrice) {
            return false;
        }
        return true;
    }

    public function validCouponForCourse($coupon)
    {
        $cart = Cart::where('course_id', '=', $coupon['course_id'])
                    ->where('user_id', '=', Auth::user()->id)
                    ->first();

        if (!isset($cart)) {
            return back()->with('fail', 'Sorry no product found in your cart for this coupon !');
        }

        if ($cart->course_id != $coupon->course_id) {
            return back()->with('fail', 'Sorry no product found in your cart for this coupon !');
        }

        $price = $cart->offer_price > 0 ? $cart->offer_price : $cart->price;
        $per = $this->calculateDiscount($price, $coupon);

        if (!$this->validateDiscountAmount($price, $per)) {
            return back()->with('fail', 'Coupon discount amount cannot be greater than total price!');
        }

        Session::put('coupanapplied', [
            'code' => $coupon->code,
            'cpnid' => $coupon->id,
            'discount' => $per,
            'msg' => "$coupon->code is applied !",
            'appliedOn' => 'course'
        ]);

        Cart::where('course_id', '=', $coupon['course_id'])
            ->where('user_id', '=', Auth::user()->id)
            ->update(['distype' => 'course', 'disamount' => $per, 'coupon_id' => $coupon->id]);

        Cart::where('course_id', '!=', $coupon['course_id'])
            ->where('user_id', '=', Auth::user()->id)
            ->update(['distype' => NULL, 'disamount' => NULL]);

        DB::table('coupons')->where('code', '=', $coupon['code'])->decrement('maxusage', 1);

        return back();
    }

    public function validCouponForBundle($coupon)
    {
        $cart = Cart::where('bundle_id', '=', $coupon['bundle_id'])
                    ->where('user_id', '=', Auth::user()->id)
                    ->first();

        if (!isset($cart)) {
            return back()->with('fail', 'Sorry no product found in your cart for this coupon !');
        }

        if ($cart->bundle_id != $coupon->bundle_id) {
            return back()->with('fail', 'Sorry no product found in your cart for this coupon !');
        }

        $price = $cart->offer_price > 0 ? $cart->offer_price : $cart->price;
        $per = $this->calculateDiscount($price, $coupon);

        if (!$this->validateDiscountAmount($price, $per)) {
            return back()->with('fail', 'Coupon discount amount cannot be greater than total price!');
        }

        Session::put('coupanapplied', [
            'code' => $coupon->code,
            'cpnid' => $coupon->id,
            'discount' => $per,
            'msg' => "$coupon->code is applied !",
            'appliedOn' => 'bundle',
            'bundle_id' => $cart->bundle_id
        ]);

        $cart->update(['distype' => 'bundle', 'disamount' => $per, 'coupon_id' => $coupon->id]);

        Cart::where('bundle_id', '!=', $coupon['bundle_id'])
            ->where('user_id', '=', Auth::user()->id)
            ->update(['distype' => NULL, 'disamount' => NULL]);

        DB::table('coupons')->where('code', '=', $coupon['code'])->decrement('maxusage', 1);

        return back();
    }

    public function validCouponForCart($coupon)
    {
        $cart = Cart::where('user_id', '=', Auth::user()->id)->get();

        if (!isset($cart) || $cart->isEmpty()) {
            return back()->with('fail', 'Your cart is empty!');
        }

        $total = 0;
        foreach ($cart as $c) {
            $total += ($c->offer_price != 0) ? $c->offer_price : $c->price;
        }

        if ($coupon->minamount != 0 && $total < $coupon->minamount) {
            return back()->with('fail', 'For Apply this coupon your cart total should be ' . $coupon->minamount . ' or greater !');
        }

        $totaldiscount = 0;
        foreach ($cart as $c) {
            $itemPrice = ($c->offer_price != 0) ? $c->offer_price : $c->price;
            $per = ($coupon->distype == 'per') 
                ? ($itemPrice * $coupon->amount / 100)
                : ($coupon->amount / count($cart));
            
            $totaldiscount += $per;
        }

        if (!$this->validateDiscountAmount($total, $totaldiscount)) {
            return back()->with('fail', 'Coupon discount amount cannot be greater than total price!');
        }

        foreach ($cart as $c) {
            $itemPrice = ($c->offer_price != 0) ? $c->offer_price : $c->price;
            $per = ($coupon->distype == 'per') 
                ? ($itemPrice * $coupon->amount / 100)
                : ($coupon->amount / count($cart));

            Cart::where('id', '=', $c->id)
                ->update(['distype' => 'cart', 'disamount' => $per, 'coupon_id' => $coupon->id]);
        }

        Session::put('coupanapplied', [
            'code' => $coupon->code,
            'cpnid' => $coupon->id,
            'discount' => $totaldiscount,
            'msg' => "$coupon->code Applied Successfully !",
            'appliedOn' => 'cart'
        ]);

        DB::table('coupons')->where('code', '=', $coupon['code'])->decrement('maxusage', 1);

        return back();
    }

    public function validCouponForCategory($coupon)
    {
        $cart = Cart::where('user_id', '=', Auth::user()->id)
                    ->where('course_id', '!=', NULL)
                    ->get();

        $catcart = collect();
        foreach ($cart as $row) {
            if ($row->courses->category->id == $coupon->category_id) {
                $catcart->push($row);
            }
        }

        if ($catcart->isEmpty()) {
            return back()->with('fail', 'Sorry no matching product found in your cart for this coupon !');
        }

        $total = 0;
        foreach ($catcart as $row) {
            $total += ($row->offer_price != 0) ? $row->offer_price : $row->price;
        }

        if ($coupon->minamount != 0 && $total < $coupon->minamount) {
            Cart::where('user_id', Auth::user()->id)
                ->update(['distype' => NULL, 'disamount' => NULL]);
            return back()->with('fail', 'For Apply this coupon your cart total should be ' . $coupon->minamount . ' or greater !');
        }

        $totaldiscount = 0;
        foreach ($catcart as $c) {
            $itemPrice = ($c->offer_price != 0) ? $c->offer_price : $c->price;
            $per = ($coupon->distype == 'per') 
                ? ($itemPrice * $coupon->amount / 100)
                : ($coupon->amount / count($catcart));
            
            $totaldiscount += $per;
        }

        if (!$this->validateDiscountAmount($total, $totaldiscount)) {
            return back()->with('fail', 'Coupon discount amount cannot be greater than total price!');
        }

        foreach ($catcart as $c) {
            $itemPrice = ($c->offer_price != 0) ? $c->offer_price : $c->price;
            $per = ($coupon->distype == 'per') 
                ? ($itemPrice * $coupon->amount / 100)
                : ($coupon->amount / count($catcart));

            Cart::where('id', '=', $c->id)
                ->where('user_id', Auth::user()->id)
                ->update(['distype' => 'category', 'disamount' => $per, 'coupon_id' => $coupon->id]);

            Cart::where('category_id', '!=', $c->courses->category['id'])
                ->where('user_id', '=', Auth::user()->id)
                ->update(['distype' => NULL, 'disamount' => NULL]);
        }

        Session::put('coupanapplied', [
            'code' => $coupon->code,
            'cpnid' => $coupon->id,
            'discount' => $totaldiscount,
            'msg' => "$coupon->code Applied Successfully !",
            'appliedOn' => 'category'
        ]);

        DB::table('coupons')->where('code', '=', $coupon['code'])->decrement('maxusage', 1);

        return back();
    }

    public function remove($cpnid)
    {
        Session::forget('coupanapplied');

        DB::table('coupons')->where('id', $cpnid)->increment('maxusage', 1);

        Cart::where('user_id', '=', Auth::user()->id)
            ->update(['distype' => NULL, 'disamount' => NULL]);
            
        return back()->with('fail', 'Coupon Removed !');
    }
}