<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Setting;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Http;
use GuzzleHttp\Client;
use App\Openai;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\log;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ChatgptController extends Controller
{


    protected function getApiKey()
    {
        try {
            $settings = Setting::first();
            if (!$settings || empty($settings->api_key)) {
                throw new \Exception('API key not configured');
            }

            // First try to decrypt normally
            try {
                $apiKey = decrypt($settings->api_key);
            } catch (\Illuminate\Contracts\Encryption\DecryptException $e) {
                // If decryption fails, check if it's a plain API key
                if (substr($settings->api_key, 0, 2) === 'sk-') {
                    $apiKey = $settings->api_key;
                } else {
                    throw new \Exception('Invalid API key format');
                }
            }

            // Validate the API key format
            if (!preg_match('/^sk-[a-zA-Z0-9]+$/', $apiKey)) {
                throw new \Exception('Invalid API key format');
            }

            return $apiKey;

        } catch (\Exception $e) {
            Log::error('API key retrieval failed: ' . $e->getMessage());
            throw new \Exception('Failed to retrieve valid API key');
        }
    }



    public function text(Request $request)
    {
        if (config('app.demolock') == 1) {
            return response()->json([
                'status' => false,
                'message' => "Demo lock is enabled",
                'type' => 'error'
            ]);
        }

        try {

            $apiKey = $this->getApiKey();

            // Validate incoming request
            $validator = Validator::make($request->all(), [
                'service' => 'required|string',
                'language' => 'required|string',
                'keyword' => 'required|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => $validator->errors()->first(),
                    'type' => 'error'
                ]);
            }

            $prompt = "Generate a professional profile for a {$request->service} in {$request->language} based on the specific details of {$request->keyword}.";
            
            // Make OpenAI API request
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type' => 'application/json',
            ])->post('https://api.openai.com/v1/chat/completions', [
                'model' => 'gpt-3.5-turbo',
                'messages' => [
                    ['role' => 'user', 'content' => $prompt]
                ],
                'temperature' => 0.7,
                'max_tokens' => 300,
            ]);

            if ($response->failed()) {
                $errorMessage = $response->json()['error']['message'] ?? 'API request failed';
                throw new \Exception($errorMessage);
            }

            $output = $response->json()['choices'][0]['message']['content'] ?? null;
            if (!$output) {
                throw new \Exception('No content generated');
            }

            // Save to database
            Openai::create([
                'generate' => 'Professional Profile Generation',
                'user_id' => Auth::id(),
                'prompt' => $prompt,
                'response' => json_encode($output),
            ]);

            return $this->textoutput($output);

        } catch (\Exception $e) {
            Log::error('Text generation error: ' . $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Error: ' . $e->getMessage(),
                'type' => 'error'
            ]);
        }
    }




    public function image(Request $request)
    {
        $apiKey = $this->getApiKey();

        if (config('app.demolock') == 1) {
            return response()->json([
                'status' => false,
                'message' => "Demo lock is disabled",
                'type' => 'error'
            ]);
        }

        try {
            $validator = Validator::make($request->all(), [
                'description' => 'required|string',
                'image_number_of_images' => 'required|integer|min:1|max:10',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => $validator->errors()->first(),
                    'type' => 'error'
                ]);
            }


            $client = new Client();
            $response = $client->post('https://api.openai.com/v1/images/generations', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $apiKey,
                    'Content-Type' => 'application/json',
                ],
                'json' => [
                    'model' => 'dall-e-3',
                    'prompt' => $request->description,
                    'n' => (int)$request->image_number_of_images,
                    'size' => '1024x1024',
                ]
            ]);

            $result = json_decode($response->getBody(), true);
            
            if (empty($result['data'])) {
                throw new \Exception('No images generated');
            }

            $imageUrl = $result['data'][0]['url'];
            
            // Save images and create records
            foreach ($result['data'] as $value) {
                $contents = file_get_contents($value['url']);
                $nameOfImage = Str::random(12) . '-' . Str::slug($request->description) . '.png';
                $imagePath = public_path('images/openai/' . $nameOfImage);
                
                if (!file_put_contents($imagePath, $contents)) {
                    throw new \Exception('Failed to save image');
                }

                $publicImageUrl = asset('/images/openai/' . $nameOfImage);

                Openai::create([
                    'generate' => 'Image Generate',
                    'user_id' => Auth::id(),
                    'prompt' => $request->description,
                    'response' => $publicImageUrl
                ]);
            }

            return $this->imagegenerate($imageUrl);

        } catch (\Exception $e) {
            Log::error('Image generation error: ' . $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Error: ' . $e->getMessage(),
                'type' => 'error'
            ]);
        }
    }

    public function textoutput($output)
    {
        $data = $output;
        $html = view('admin.openai.output', compact('data'))->render();
        return response()->json(compact('html'));
    }

    public function imagegenerate($imageUrl)
    {
        $response = view('admin.openai.image', compact('imageUrl'))->render();
        $status = 'True';
        return response()->json(compact('response', 'status'));
    }

    public function useropenai()
    {
        if (Auth::check()) {
            if (Auth::user()->role == 'admin') {
                $openai = Openai::orderBy('created_at', 'desc')->get();
            } else {
                $openai = Openai::where('user_id', Auth::id())->orderBy('created_at', 'desc')->get();
            }
        } else {
            return redirect()->route('login');
        }

        return view('admin.openai.user', compact('openai'));
    }

    public function delete($id)
    {
        DB::table('openais')->where('id', $id)->delete();
        Session::flash('delete', trans('Deleted Successfully'));
        return redirect('user/openai');
    }

    public function bulk_delete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'checked' => 'required',
        ]);

        if ($validator->fails()) {
            return back()->with('warning', 'At least one item is required to be checked');
        } else {
            Openai::whereIn('id', $request->checked)->delete();
            Session::flash('success', trans('Deleted Successfully'));
            return redirect()->back();
        }
    }
}
